#ifndef CAPSTONE_SPARC_H
#define CAPSTONE_SPARC_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2014-2015 */

#ifdef __cplusplus
extern "C" {
#endif

#include "platform.h"
#include "cs_operand.h"

// GCC SPARC toolchain has a default macro called "sparc" which breaks
// compilation
#undef sparc

#ifdef _MSC_VER
#pragma warning(disable:4201)
#endif

/// Enums corresponding to Sparc condition codes, both icc's and fcc's.
typedef enum sparc_cc {
	// Integer condition codes
	SPARC_CC_ICC_A = 8,    // Always
	SPARC_CC_ICC_N = 0,    // Never
	SPARC_CC_ICC_NE = 9,   // Not Equal
	SPARC_CC_ICC_E = 1,    // Equal
	SPARC_CC_ICC_G = 10,   // Greater
	SPARC_CC_ICC_LE = 2,   // Less or Equal
	SPARC_CC_ICC_GE = 11,  // Greater or Equal
	SPARC_CC_ICC_L = 3,    // Less
	SPARC_CC_ICC_GU = 12,  // Greater Unsigned
	SPARC_CC_ICC_LEU = 4,  // Less or Equal Unsigned
	SPARC_CC_ICC_CC = 13,  // Carry Clear/Great or Equal Unsigned
	SPARC_CC_ICC_CS = 5,   // Carry Set/Less Unsigned
	SPARC_CC_ICC_POS = 14, // Positive
	SPARC_CC_ICC_NEG = 6,  // Negative
	SPARC_CC_ICC_VC = 15,  // Overflow Clear
	SPARC_CC_ICC_VS = 7,   // Overflow Set

	SPARC_CC_FCC_BEGIN = 16,
	SPARC_CC_FCC_A = 8 + SPARC_CC_FCC_BEGIN,    // Always
	SPARC_CC_FCC_N = 0 + SPARC_CC_FCC_BEGIN,    // Never
	SPARC_CC_FCC_U = 7 + SPARC_CC_FCC_BEGIN,    // Unordered
	SPARC_CC_FCC_G = 6 + SPARC_CC_FCC_BEGIN,    // Greater
	SPARC_CC_FCC_UG = 5 + SPARC_CC_FCC_BEGIN,   // Unordered or Greater
	SPARC_CC_FCC_L = 4 + SPARC_CC_FCC_BEGIN,    // Less
	SPARC_CC_FCC_UL = 3 + SPARC_CC_FCC_BEGIN,   // Unordered or Less
	SPARC_CC_FCC_LG = 2 + SPARC_CC_FCC_BEGIN,   // Less or Greater
	SPARC_CC_FCC_NE = 1 + SPARC_CC_FCC_BEGIN,   // Not Equal
	SPARC_CC_FCC_E = 9 + SPARC_CC_FCC_BEGIN,    // Equal
	SPARC_CC_FCC_UE = 10 + SPARC_CC_FCC_BEGIN,  // Unordered or Equal
	SPARC_CC_FCC_GE = 11 + SPARC_CC_FCC_BEGIN,  // Greater or Equal
	SPARC_CC_FCC_UGE = 12 + SPARC_CC_FCC_BEGIN, // Unordered or Greater or Equal
	SPARC_CC_FCC_LE = 13 + SPARC_CC_FCC_BEGIN,  // Less or Equal
	SPARC_CC_FCC_ULE = 14 + SPARC_CC_FCC_BEGIN, // Unordered or Less or Equal
	SPARC_CC_FCC_O = 15 + SPARC_CC_FCC_BEGIN,   // Ordered

	SPARC_CC_CPCC_BEGIN = 32, ///< Co-processor conditional branches
	SPARC_CC_CPCC_A = 8 + SPARC_CC_CPCC_BEGIN, // Always
	SPARC_CC_CPCC_N = 0 + SPARC_CC_CPCC_BEGIN, // Never
	SPARC_CC_CPCC_3 = 7 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_2 = 6 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_23 = 5 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_1 = 4 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_13 = 3 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_12 = 2 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_123 = 1 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_0 = 9 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_03 = 10 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_02 = 11 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_023 = 12 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_01 = 13 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_013 = 14 + SPARC_CC_CPCC_BEGIN,
	SPARC_CC_CPCC_012 = 15 + SPARC_CC_CPCC_BEGIN,

	SPARC_CC_REG_BEGIN = 48,
	SPARC_CC_REG_Z = 1 + SPARC_CC_REG_BEGIN,   // Is zero
	SPARC_CC_REG_LEZ = 2 + SPARC_CC_REG_BEGIN, // Less or equal to zero
	SPARC_CC_REG_LZ = 3 + SPARC_CC_REG_BEGIN,  // Less than zero
	SPARC_CC_REG_NZ = 5 + SPARC_CC_REG_BEGIN,  // Is not zero
	SPARC_CC_REG_GZ = 6 + SPARC_CC_REG_BEGIN,  // Greater than zero
	SPARC_CC_REG_GEZ = 7 + SPARC_CC_REG_BEGIN,  // Greater than or equal to zero

	SPARC_CC_UNDEF = 0xffff,
} sparc_cc;

inline static const char *SPARCCondCodeToString(sparc_cc CC) {
  switch (CC) {
  case SPARC_CC_ICC_A:   return "a";
  case SPARC_CC_ICC_N:   return "n";
  case SPARC_CC_ICC_NE:  return "ne";
  case SPARC_CC_ICC_E:   return "e";
  case SPARC_CC_ICC_G:   return "g";
  case SPARC_CC_ICC_LE:  return "le";
  case SPARC_CC_ICC_GE:  return "ge";
  case SPARC_CC_ICC_L:   return "l";
  case SPARC_CC_ICC_GU:  return "gu";
  case SPARC_CC_ICC_LEU: return "leu";
  case SPARC_CC_ICC_CC:  return "cc";
  case SPARC_CC_ICC_CS:  return "cs";
  case SPARC_CC_ICC_POS: return "pos";
  case SPARC_CC_ICC_NEG: return "neg";
  case SPARC_CC_ICC_VC:  return "vc";
  case SPARC_CC_ICC_VS:  return "vs";
  case SPARC_CC_FCC_A:   return "a";
  case SPARC_CC_FCC_N:   return "n";
  case SPARC_CC_FCC_U:   return "u";
  case SPARC_CC_FCC_G:   return "g";
  case SPARC_CC_FCC_UG:  return "ug";
  case SPARC_CC_FCC_L:   return "l";
  case SPARC_CC_FCC_UL:  return "ul";
  case SPARC_CC_FCC_LG:  return "lg";
  case SPARC_CC_FCC_NE:  return "ne";
  case SPARC_CC_FCC_E:   return "e";
  case SPARC_CC_FCC_UE:  return "ue";
  case SPARC_CC_FCC_GE:  return "ge";
  case SPARC_CC_FCC_UGE: return "uge";
  case SPARC_CC_FCC_LE:  return "le";
  case SPARC_CC_FCC_ULE: return "ule";
  case SPARC_CC_FCC_O:   return "o";
  case SPARC_CC_CPCC_A:   return "a";
  case SPARC_CC_CPCC_N:   return "n";
  case SPARC_CC_CPCC_3:   return "3";
  case SPARC_CC_CPCC_2:   return "2";
  case SPARC_CC_CPCC_23:  return "23";
  case SPARC_CC_CPCC_1:   return "1";
  case SPARC_CC_CPCC_13:  return "13";
  case SPARC_CC_CPCC_12:  return "12";
  case SPARC_CC_CPCC_123: return "123";
  case SPARC_CC_CPCC_0:   return "0";
  case SPARC_CC_CPCC_03:  return "03";
  case SPARC_CC_CPCC_02:  return "02";
  case SPARC_CC_CPCC_023: return "023";
  case SPARC_CC_CPCC_01:  return "01";
  case SPARC_CC_CPCC_013: return "013";
  case SPARC_CC_CPCC_012: return "012";
  case SPARC_CC_REG_BEGIN:
  case SPARC_CC_UNDEF:
    return "invalid";
  case SPARC_CC_REG_Z:
    return "z";
  case SPARC_CC_REG_LEZ:
    return "lez";
  case SPARC_CC_REG_LZ:
    return "lz";
  case SPARC_CC_REG_NZ:
    return "nz";
  case SPARC_CC_REG_GZ:
    return "gz";
  case SPARC_CC_REG_GEZ:
    return "gez";
  }
  return "invalid";
}

/// The cc field values encoded in the cc0, cc1, cc2 bits or opf_cc.
/// They are equivalent to ISA Sparc v9 - Table 41 - Chapter E.2 Tables - Page 281.
typedef enum {
	SPARC_CC_FIELD_FCC0 = 0,
	SPARC_CC_FIELD_FCC1 = 1,
	SPARC_CC_FIELD_FCC2 = 2,
	SPARC_CC_FIELD_FCC3 = 3,
	SPARC_CC_FIELD_ICC = 4,
	SPARC_CC_FIELD_RESERVED_I = 5,
	SPARC_CC_FIELD_XCC = 6,
	SPARC_CC_FIELD_RESERVED_II = 7,
	SPARC_CC_FIELD_NONE = 0xff,
} sparc_cc_field;

/// The instruction formats.
/// They are only loosely equivalent to the one in the ISA.
/// Because in LLVM they are not defined one to one.
typedef enum {
	// generated content <SparcGenCSInsnFormatsEnum.inc> begin
	// clang-format off

	SPARC_INSN_FORM_ASMPSEUDOINST,
	SPARC_INSN_FORM_F3_2,
	SPARC_INSN_FORM_F3_1_ASI,
	SPARC_INSN_FORM_F3_3,
	SPARC_INSN_FORM_F2_2,
	SPARC_INSN_FORM_F2_3,
	SPARC_INSN_FORM_F2_4,
	SPARC_INSN_FORM_F1,
	SPARC_INSN_FORM_F3_3U,
	SPARC_INSN_FORM_F4_3,
	SPARC_INSN_FORM_F4_4R,
	SPARC_INSN_FORM_F4_2,
	SPARC_INSN_FORM_F4_1,
	SPARC_INSN_FORM_F4_4I,
	SPARC_INSN_FORM_F2_1,
	SPARC_INSN_FORM_F3_SI,
	SPARC_INSN_FORM_F3_SR,
	SPARC_INSN_FORM_TRAPSP,
	SPARC_INSN_FORM_F3_3C,

	// clang-format on
	// generated content <SparcGenCSInsnFormatsEnum.inc> end
} sparc_insn_form;

/// Branch hint
typedef enum sparc_hint {
	SPARC_HINT_INVALID = 0,	///< no hint
	SPARC_HINT_A	= 1 << 0,	///< annul delay slot instruction
	SPARC_HINT_PT	= 1 << 1,	///< branch taken
	SPARC_HINT_PN	= 1 << 2,	///< branch NOT taken
	SPARC_HINT_A_PN	= SPARC_HINT_A | SPARC_HINT_PN,
	SPARC_HINT_A_PT	= SPARC_HINT_A | SPARC_HINT_PT,
} sparc_hint;

/// Operand type for instruction's operands
typedef enum sparc_op_type {
	SPARC_OP_INVALID = CS_OP_INVALID, ///< = CS_OP_INVALID (Uninitialized).
	SPARC_OP_REG = CS_OP_REG, ///< = CS_OP_REG (Register operand).
	SPARC_OP_IMM = CS_OP_IMM, ///< = CS_OP_IMM (Immediate operand).
	SPARC_OP_MEM = CS_OP_MEM, ///< = CS_OP_MEM (Memory operand).
	SPARC_OP_MEMBAR_TAG = CS_OP_SPECIAL + 0,
	SPARC_OP_ASI = CS_OP_SPECIAL + 1,
} sparc_op_type;

/// SPARC registers
typedef enum sparc_reg {
	// generated content <SparcGenCSRegEnum.inc> begin
	// clang-format off

	SPARC_REG_INVALID = 0,
	SPARC_REG_CANRESTORE = 1,
	SPARC_REG_CANSAVE = 2,
	SPARC_REG_CLEANWIN = 3,
	SPARC_REG_CPQ = 4,
	SPARC_REG_CPSR = 5,
	SPARC_REG_CWP = 6,
	SPARC_REG_FQ = 7,
	SPARC_REG_FSR = 8,
	SPARC_REG_GL = 9,
	SPARC_REG_ICC = 10,
	SPARC_REG_OTHERWIN = 11,
	SPARC_REG_PIL = 12,
	SPARC_REG_PSR = 13,
	SPARC_REG_PSTATE = 14,
	SPARC_REG_TBA = 15,
	SPARC_REG_TBR = 16,
	SPARC_REG_TICK = 17,
	SPARC_REG_TL = 18,
	SPARC_REG_TNPC = 19,
	SPARC_REG_TPC = 20,
	SPARC_REG_TSTATE = 21,
	SPARC_REG_TT = 22,
	SPARC_REG_VER = 23,
	SPARC_REG_WIM = 24,
	SPARC_REG_WSTATE = 25,
	SPARC_REG_Y = 26,
	SPARC_REG_ASR1 = 27,
	SPARC_REG_ASR2 = 28,
	SPARC_REG_ASR3 = 29,
	SPARC_REG_ASR4 = 30,
	SPARC_REG_ASR5 = 31,
	SPARC_REG_ASR6 = 32,
	SPARC_REG_ASR7 = 33,
	SPARC_REG_ASR8 = 34,
	SPARC_REG_ASR9 = 35,
	SPARC_REG_ASR10 = 36,
	SPARC_REG_ASR11 = 37,
	SPARC_REG_ASR12 = 38,
	SPARC_REG_ASR13 = 39,
	SPARC_REG_ASR14 = 40,
	SPARC_REG_ASR15 = 41,
	SPARC_REG_ASR16 = 42,
	SPARC_REG_ASR17 = 43,
	SPARC_REG_ASR18 = 44,
	SPARC_REG_ASR19 = 45,
	SPARC_REG_ASR20 = 46,
	SPARC_REG_ASR21 = 47,
	SPARC_REG_ASR22 = 48,
	SPARC_REG_ASR23 = 49,
	SPARC_REG_ASR24 = 50,
	SPARC_REG_ASR25 = 51,
	SPARC_REG_ASR26 = 52,
	SPARC_REG_ASR27 = 53,
	SPARC_REG_ASR28 = 54,
	SPARC_REG_ASR29 = 55,
	SPARC_REG_ASR30 = 56,
	SPARC_REG_ASR31 = 57,
	SPARC_REG_C0 = 58,
	SPARC_REG_C1 = 59,
	SPARC_REG_C2 = 60,
	SPARC_REG_C3 = 61,
	SPARC_REG_C4 = 62,
	SPARC_REG_C5 = 63,
	SPARC_REG_C6 = 64,
	SPARC_REG_C7 = 65,
	SPARC_REG_C8 = 66,
	SPARC_REG_C9 = 67,
	SPARC_REG_C10 = 68,
	SPARC_REG_C11 = 69,
	SPARC_REG_C12 = 70,
	SPARC_REG_C13 = 71,
	SPARC_REG_C14 = 72,
	SPARC_REG_C15 = 73,
	SPARC_REG_C16 = 74,
	SPARC_REG_C17 = 75,
	SPARC_REG_C18 = 76,
	SPARC_REG_C19 = 77,
	SPARC_REG_C20 = 78,
	SPARC_REG_C21 = 79,
	SPARC_REG_C22 = 80,
	SPARC_REG_C23 = 81,
	SPARC_REG_C24 = 82,
	SPARC_REG_C25 = 83,
	SPARC_REG_C26 = 84,
	SPARC_REG_C27 = 85,
	SPARC_REG_C28 = 86,
	SPARC_REG_C29 = 87,
	SPARC_REG_C30 = 88,
	SPARC_REG_C31 = 89,
	SPARC_REG_D0 = 90,
	SPARC_REG_D1 = 91,
	SPARC_REG_D2 = 92,
	SPARC_REG_D3 = 93,
	SPARC_REG_D4 = 94,
	SPARC_REG_D5 = 95,
	SPARC_REG_D6 = 96,
	SPARC_REG_D7 = 97,
	SPARC_REG_D8 = 98,
	SPARC_REG_D9 = 99,
	SPARC_REG_D10 = 100,
	SPARC_REG_D11 = 101,
	SPARC_REG_D12 = 102,
	SPARC_REG_D13 = 103,
	SPARC_REG_D14 = 104,
	SPARC_REG_D15 = 105,
	SPARC_REG_D16 = 106,
	SPARC_REG_D17 = 107,
	SPARC_REG_D18 = 108,
	SPARC_REG_D19 = 109,
	SPARC_REG_D20 = 110,
	SPARC_REG_D21 = 111,
	SPARC_REG_D22 = 112,
	SPARC_REG_D23 = 113,
	SPARC_REG_D24 = 114,
	SPARC_REG_D25 = 115,
	SPARC_REG_D26 = 116,
	SPARC_REG_D27 = 117,
	SPARC_REG_D28 = 118,
	SPARC_REG_D29 = 119,
	SPARC_REG_D30 = 120,
	SPARC_REG_D31 = 121,
	SPARC_REG_F0 = 122,
	SPARC_REG_F1 = 123,
	SPARC_REG_F2 = 124,
	SPARC_REG_F3 = 125,
	SPARC_REG_F4 = 126,
	SPARC_REG_F5 = 127,
	SPARC_REG_F6 = 128,
	SPARC_REG_F7 = 129,
	SPARC_REG_F8 = 130,
	SPARC_REG_F9 = 131,
	SPARC_REG_F10 = 132,
	SPARC_REG_F11 = 133,
	SPARC_REG_F12 = 134,
	SPARC_REG_F13 = 135,
	SPARC_REG_F14 = 136,
	SPARC_REG_F15 = 137,
	SPARC_REG_F16 = 138,
	SPARC_REG_F17 = 139,
	SPARC_REG_F18 = 140,
	SPARC_REG_F19 = 141,
	SPARC_REG_F20 = 142,
	SPARC_REG_F21 = 143,
	SPARC_REG_F22 = 144,
	SPARC_REG_F23 = 145,
	SPARC_REG_F24 = 146,
	SPARC_REG_F25 = 147,
	SPARC_REG_F26 = 148,
	SPARC_REG_F27 = 149,
	SPARC_REG_F28 = 150,
	SPARC_REG_F29 = 151,
	SPARC_REG_F30 = 152,
	SPARC_REG_F31 = 153,
	SPARC_REG_FCC0 = 154,
	SPARC_REG_FCC1 = 155,
	SPARC_REG_FCC2 = 156,
	SPARC_REG_FCC3 = 157,
	SPARC_REG_G0 = 158,
	SPARC_REG_G1 = 159,
	SPARC_REG_G2 = 160,
	SPARC_REG_G3 = 161,
	SPARC_REG_G4 = 162,
	SPARC_REG_G5 = 163,
	SPARC_REG_G6 = 164,
	SPARC_REG_G7 = 165,
	SPARC_REG_I0 = 166,
	SPARC_REG_I1 = 167,
	SPARC_REG_I2 = 168,
	SPARC_REG_I3 = 169,
	SPARC_REG_I4 = 170,
	SPARC_REG_I5 = 171,
	SPARC_REG_I6 = 172,
	SPARC_REG_I7 = 173,
	SPARC_REG_L0 = 174,
	SPARC_REG_L1 = 175,
	SPARC_REG_L2 = 176,
	SPARC_REG_L3 = 177,
	SPARC_REG_L4 = 178,
	SPARC_REG_L5 = 179,
	SPARC_REG_L6 = 180,
	SPARC_REG_L7 = 181,
	SPARC_REG_O0 = 182,
	SPARC_REG_O1 = 183,
	SPARC_REG_O2 = 184,
	SPARC_REG_O3 = 185,
	SPARC_REG_O4 = 186,
	SPARC_REG_O5 = 187,
	SPARC_REG_O6 = 188,
	SPARC_REG_O7 = 189,
	SPARC_REG_Q0 = 190,
	SPARC_REG_Q1 = 191,
	SPARC_REG_Q2 = 192,
	SPARC_REG_Q3 = 193,
	SPARC_REG_Q4 = 194,
	SPARC_REG_Q5 = 195,
	SPARC_REG_Q6 = 196,
	SPARC_REG_Q7 = 197,
	SPARC_REG_Q8 = 198,
	SPARC_REG_Q9 = 199,
	SPARC_REG_Q10 = 200,
	SPARC_REG_Q11 = 201,
	SPARC_REG_Q12 = 202,
	SPARC_REG_Q13 = 203,
	SPARC_REG_Q14 = 204,
	SPARC_REG_Q15 = 205,
	SPARC_REG_C0_C1 = 206,
	SPARC_REG_C2_C3 = 207,
	SPARC_REG_C4_C5 = 208,
	SPARC_REG_C6_C7 = 209,
	SPARC_REG_C8_C9 = 210,
	SPARC_REG_C10_C11 = 211,
	SPARC_REG_C12_C13 = 212,
	SPARC_REG_C14_C15 = 213,
	SPARC_REG_C16_C17 = 214,
	SPARC_REG_C18_C19 = 215,
	SPARC_REG_C20_C21 = 216,
	SPARC_REG_C22_C23 = 217,
	SPARC_REG_C24_C25 = 218,
	SPARC_REG_C26_C27 = 219,
	SPARC_REG_C28_C29 = 220,
	SPARC_REG_C30_C31 = 221,
	SPARC_REG_G0_G1 = 222,
	SPARC_REG_G2_G3 = 223,
	SPARC_REG_G4_G5 = 224,
	SPARC_REG_G6_G7 = 225,
	SPARC_REG_I0_I1 = 226,
	SPARC_REG_I2_I3 = 227,
	SPARC_REG_I4_I5 = 228,
	SPARC_REG_I6_I7 = 229,
	SPARC_REG_L0_L1 = 230,
	SPARC_REG_L2_L3 = 231,
	SPARC_REG_L4_L5 = 232,
	SPARC_REG_L6_L7 = 233,
	SPARC_REG_O0_O1 = 234,
	SPARC_REG_O2_O3 = 235,
	SPARC_REG_O4_O5 = 236,
	SPARC_REG_O6_O7 = 237,
	SPARC_REG_ENDING, // 238

	// clang-format on
	// generated content <SparcGenCSRegEnum.inc> end

	// extras
	SPARC_REG_SP = SPARC_REG_O6,
	SPARC_REG_FP = SPARC_REG_I6,
} sparc_reg;

typedef enum sparc_asi {
	SPARC_ASITAG_ASI_INVALID = 0x0,
	// generated content <SparcGenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ASITag> begin
	// clang-format off

	SPARC_ASITAG_ASI_AIUP = 0x10,
	SPARC_ASITAG_ASI_AS_IF_USER_PRIMARY = 0x10,
	SPARC_ASITAG_ASI_AIUP_L = 0x18,
	SPARC_ASITAG_ASI_AS_IF_USER_PRIMARY_LITTLE = 0x18,
	SPARC_ASITAG_ASI_AIUS = 0x11,
	SPARC_ASITAG_ASI_AS_IF_USER_SECONDARY = 0x11,
	SPARC_ASITAG_ASI_AIUS_L = 0x19,
	SPARC_ASITAG_ASI_AS_IF_USER_SECONDARY_LITTLE = 0x19,
	SPARC_ASITAG_ASI_N = 0x4,
	SPARC_ASITAG_ASI_NUCLEUS = 0x4,
	SPARC_ASITAG_ASI_N_L = 0xc,
	SPARC_ASITAG_ASI_NUCLEUS_LITTLE = 0xc,
	SPARC_ASITAG_ASI_P = 0x80,
	SPARC_ASITAG_ASI_PRIMARY = 0x80,
	SPARC_ASITAG_ASI_PNF = 0x82,
	SPARC_ASITAG_ASI_PRIMARY_NOFAULT = 0x82,
	SPARC_ASITAG_ASI_PNF_L = 0x8a,
	SPARC_ASITAG_ASI_PRIMARY_NOFAULT_LITTLE = 0x8a,
	SPARC_ASITAG_ASI_P_L = 0x88,
	SPARC_ASITAG_ASI_PRIMARY_LITTLE = 0x88,
	SPARC_ASITAG_ASI_S = 0x81,
	SPARC_ASITAG_ASI_SECONDARY = 0x81,
	SPARC_ASITAG_ASI_SNF = 0x83,
	SPARC_ASITAG_ASI_SECONDARY_NOFAULT = 0x83,
	SPARC_ASITAG_ASI_SNF_L = 0x8b,
	SPARC_ASITAG_ASI_SECONDARY_NOFAULT_LITTLE = 0x8b,
	SPARC_ASITAG_ASI_S_L = 0x89,
	SPARC_ASITAG_ASI_SECONDARY_LITTLE = 0x89,

	// clang-format on
	// generated content <SparcGenCSSystemOperandsEnum.inc:GET_ENUM_VALUES_ASITag> end
} sparc_asi;

typedef enum sparc_membar_tag {
	SPARC_MEMBAR_TAG_NONE = 0,
	SPARC_MEMBAR_TAG_LOADLOAD = 1,
	SPARC_MEMBAR_TAG_STORELOAD = 1 << 1,
	SPARC_MEMBAR_TAG_LOADSTORE = 1 << 2,
	SPARC_MEMBAR_TAG_STORESTORE = 1 << 3,
	SPARC_MEMBAR_TAG_LOOKASIDE = 1 << 4,
	SPARC_MEMBAR_TAG_MEMISSUE = 1 << 5,
	SPARC_MEMBAR_TAG_SYNC = 1 << 6,
} sparc_membar_tag;

/// Instruction's operand referring to memory
/// This is associated with SPARC_OP_MEM operand type above
typedef struct sparc_op_mem {
	sparc_reg base;		///< base register, can be safely interpreted as
				///< a value of type `sparc_reg`, but it is only
				///< one byte wide
	sparc_reg index;		///< index register, same conditions apply here
	int32_t disp;		///< displacement/offset value
} sparc_op_mem;

/// Instruction operand
typedef struct cs_sparc_op {
	sparc_op_type type;	///< operand type
	union {
		sparc_reg reg;	///< register value for REG operand
		int64_t imm;		///< immediate value for IMM operand
		sparc_op_mem mem;		///< base/disp value for MEM operand
		sparc_membar_tag membar_tag; ///< Tag of the MEMBAR instruction.
		sparc_asi asi; ///< Address space identifier.
	};
	cs_ac_type access; ///< The way the operand is accessed.
} cs_sparc_op;

#define NUM_SPARC_OPS 6

/// Instruction structure
typedef struct cs_sparc {
	sparc_cc cc;	///< Condition code: only SPARC_CC_ICC_* are set here.
	sparc_cc_field cc_field; ///< The CC field the instruction uses.
	sparc_hint hint;	///< branch hint: encoding as bitwise OR of sparc_hint.
	sparc_insn_form format; ///< The instruction format.
	/// Number of operands of this instruction,
	/// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_sparc_op operands[NUM_SPARC_OPS]; ///< operands for this instruction.
} cs_sparc;

typedef struct {
	sparc_insn_form form; ///< The instruction format.
} sparc_suppl_info;

/// SPARC instruction
typedef enum sparc_insn {
	// generated content <SparcGenCSInsnEnum.inc> begin
	// clang-format off

	SPARC_INS_INVALID,
	SPARC_INS_SET,
	SPARC_INS_SETX,
	SPARC_INS_ADDCC,
	SPARC_INS_ADDX,
	SPARC_INS_ADDXCC,
	SPARC_INS_ADDXC,
	SPARC_INS_ADDXCCC,
	SPARC_INS_ADD,
	SPARC_INS_ALIGNADDR,
	SPARC_INS_ALIGNADDRL,
	SPARC_INS_ANDCC,
	SPARC_INS_ANDNCC,
	SPARC_INS_ANDN,
	SPARC_INS_AND,
	SPARC_INS_ARRAY16,
	SPARC_INS_ARRAY32,
	SPARC_INS_ARRAY8,
	SPARC_INS_B,
	SPARC_INS_BMASK,
	SPARC_INS_FB,
	SPARC_INS_BR,
	SPARC_INS_BSHUFFLE,
	SPARC_INS_CALL,
	SPARC_INS_CASA,
	SPARC_INS_CASXA,
	SPARC_INS_CB,
	SPARC_INS_CMASK16,
	SPARC_INS_CMASK32,
	SPARC_INS_CMASK8,
	SPARC_INS_DONE,
	SPARC_INS_EDGE16,
	SPARC_INS_EDGE16L,
	SPARC_INS_EDGE16LN,
	SPARC_INS_EDGE16N,
	SPARC_INS_EDGE32,
	SPARC_INS_EDGE32L,
	SPARC_INS_EDGE32LN,
	SPARC_INS_EDGE32N,
	SPARC_INS_EDGE8,
	SPARC_INS_EDGE8L,
	SPARC_INS_EDGE8LN,
	SPARC_INS_EDGE8N,
	SPARC_INS_FABSD,
	SPARC_INS_FABSQ,
	SPARC_INS_FABSS,
	SPARC_INS_FADDD,
	SPARC_INS_FADDQ,
	SPARC_INS_FADDS,
	SPARC_INS_FALIGNDATA,
	SPARC_INS_FAND,
	SPARC_INS_FANDNOT1,
	SPARC_INS_FANDNOT1S,
	SPARC_INS_FANDNOT2,
	SPARC_INS_FANDNOT2S,
	SPARC_INS_FANDS,
	SPARC_INS_FCHKSM16,
	SPARC_INS_FCMPEQ16,
	SPARC_INS_FCMPEQ32,
	SPARC_INS_FCMPGT16,
	SPARC_INS_FCMPGT32,
	SPARC_INS_FCMPLE16,
	SPARC_INS_FCMPLE32,
	SPARC_INS_FCMPNE16,
	SPARC_INS_FCMPNE32,
	SPARC_INS_FDIVD,
	SPARC_INS_FDIVQ,
	SPARC_INS_FDIVS,
	SPARC_INS_FDMULQ,
	SPARC_INS_FDTOI,
	SPARC_INS_FDTOQ,
	SPARC_INS_FDTOS,
	SPARC_INS_FDTOX,
	SPARC_INS_FEXPAND,
	SPARC_INS_FHADDD,
	SPARC_INS_FHADDS,
	SPARC_INS_FHSUBD,
	SPARC_INS_FHSUBS,
	SPARC_INS_FITOD,
	SPARC_INS_FITOQ,
	SPARC_INS_FITOS,
	SPARC_INS_FLCMPD,
	SPARC_INS_FLCMPS,
	SPARC_INS_FLUSH,
	SPARC_INS_FLUSHW,
	SPARC_INS_FMEAN16,
	SPARC_INS_FMOVD,
	SPARC_INS_FMOVQ,
	SPARC_INS_FMOVRD,
	SPARC_INS_FMOVRQ,
	SPARC_INS_FMOVRS,
	SPARC_INS_FMOVS,
	SPARC_INS_FMUL8SUX16,
	SPARC_INS_FMUL8ULX16,
	SPARC_INS_FMUL8X16,
	SPARC_INS_FMUL8X16AL,
	SPARC_INS_FMUL8X16AU,
	SPARC_INS_FMULD,
	SPARC_INS_FMULD8SUX16,
	SPARC_INS_FMULD8ULX16,
	SPARC_INS_FMULQ,
	SPARC_INS_FMULS,
	SPARC_INS_FNADDD,
	SPARC_INS_FNADDS,
	SPARC_INS_FNAND,
	SPARC_INS_FNANDS,
	SPARC_INS_FNEGD,
	SPARC_INS_FNEGQ,
	SPARC_INS_FNEGS,
	SPARC_INS_FNHADDD,
	SPARC_INS_FNHADDS,
	SPARC_INS_FNOR,
	SPARC_INS_FNORS,
	SPARC_INS_FNOT1,
	SPARC_INS_FNOT1S,
	SPARC_INS_FNOT2,
	SPARC_INS_FNOT2S,
	SPARC_INS_FONE,
	SPARC_INS_FONES,
	SPARC_INS_FOR,
	SPARC_INS_FORNOT1,
	SPARC_INS_FORNOT1S,
	SPARC_INS_FORNOT2,
	SPARC_INS_FORNOT2S,
	SPARC_INS_FORS,
	SPARC_INS_FPACK16,
	SPARC_INS_FPACK32,
	SPARC_INS_FPACKFIX,
	SPARC_INS_FPADD16,
	SPARC_INS_FPADD16S,
	SPARC_INS_FPADD32,
	SPARC_INS_FPADD32S,
	SPARC_INS_FPADD64,
	SPARC_INS_FPMERGE,
	SPARC_INS_FPSUB16,
	SPARC_INS_FPSUB16S,
	SPARC_INS_FPSUB32,
	SPARC_INS_FPSUB32S,
	SPARC_INS_FQTOD,
	SPARC_INS_FQTOI,
	SPARC_INS_FQTOS,
	SPARC_INS_FQTOX,
	SPARC_INS_FSLAS16,
	SPARC_INS_FSLAS32,
	SPARC_INS_FSLL16,
	SPARC_INS_FSLL32,
	SPARC_INS_FSMULD,
	SPARC_INS_FSQRTD,
	SPARC_INS_FSQRTQ,
	SPARC_INS_FSQRTS,
	SPARC_INS_FSRA16,
	SPARC_INS_FSRA32,
	SPARC_INS_FSRC1,
	SPARC_INS_FSRC1S,
	SPARC_INS_FSRC2,
	SPARC_INS_FSRC2S,
	SPARC_INS_FSRL16,
	SPARC_INS_FSRL32,
	SPARC_INS_FSTOD,
	SPARC_INS_FSTOI,
	SPARC_INS_FSTOQ,
	SPARC_INS_FSTOX,
	SPARC_INS_FSUBD,
	SPARC_INS_FSUBQ,
	SPARC_INS_FSUBS,
	SPARC_INS_FXNOR,
	SPARC_INS_FXNORS,
	SPARC_INS_FXOR,
	SPARC_INS_FXORS,
	SPARC_INS_FXTOD,
	SPARC_INS_FXTOQ,
	SPARC_INS_FXTOS,
	SPARC_INS_FZERO,
	SPARC_INS_FZEROS,
	SPARC_INS_LDX,
	SPARC_INS_LD,
	SPARC_INS_JMPL,
	SPARC_INS_LDA,
	SPARC_INS_LDDA,
	SPARC_INS_LDD,
	SPARC_INS_LDQA,
	SPARC_INS_LDQ,
	SPARC_INS_LDSBA,
	SPARC_INS_LDSB,
	SPARC_INS_LDSHA,
	SPARC_INS_LDSH,
	SPARC_INS_LDSTUBA,
	SPARC_INS_LDSTUB,
	SPARC_INS_LDSWA,
	SPARC_INS_LDSW,
	SPARC_INS_LDUBA,
	SPARC_INS_LDUB,
	SPARC_INS_LDUHA,
	SPARC_INS_LDUH,
	SPARC_INS_LDXA,
	SPARC_INS_LZCNT,
	SPARC_INS_MEMBAR,
	SPARC_INS_MOVDTOX,
	SPARC_INS_MOV,
	SPARC_INS_MOVR,
	SPARC_INS_MOVSTOSW,
	SPARC_INS_MOVSTOUW,
	SPARC_INS_MULSCC,
	SPARC_INS_MULX,
	SPARC_INS_NOP,
	SPARC_INS_ORCC,
	SPARC_INS_ORNCC,
	SPARC_INS_ORN,
	SPARC_INS_OR,
	SPARC_INS_PDIST,
	SPARC_INS_PDISTN,
	SPARC_INS_POPC,
	SPARC_INS_PREFETCH,
	SPARC_INS_PWR,
	SPARC_INS_RD,
	SPARC_INS_RDPR,
	SPARC_INS_RESTORED,
	SPARC_INS_RESTORE,
	SPARC_INS_RETRY,
	SPARC_INS_RETT,
	SPARC_INS_SAVED,
	SPARC_INS_SAVE,
	SPARC_INS_SDIVCC,
	SPARC_INS_SDIVX,
	SPARC_INS_SDIV,
	SPARC_INS_SETHI,
	SPARC_INS_SHUTDOWN,
	SPARC_INS_SIAM,
	SPARC_INS_SIR,
	SPARC_INS_SLLX,
	SPARC_INS_SLL,
	SPARC_INS_SMAC,
	SPARC_INS_SMULCC,
	SPARC_INS_SMUL,
	SPARC_INS_SRAX,
	SPARC_INS_SRA,
	SPARC_INS_SRLX,
	SPARC_INS_SRL,
	SPARC_INS_STA,
	SPARC_INS_STBAR,
	SPARC_INS_STBA,
	SPARC_INS_STB,
	SPARC_INS_ST,
	SPARC_INS_STDA,
	SPARC_INS_STD,
	SPARC_INS_STHA,
	SPARC_INS_STH,
	SPARC_INS_STQA,
	SPARC_INS_STQ,
	SPARC_INS_STXA,
	SPARC_INS_STX,
	SPARC_INS_SUBCC,
	SPARC_INS_SUBX,
	SPARC_INS_SUBXCC,
	SPARC_INS_SUB,
	SPARC_INS_SWAPA,
	SPARC_INS_SWAP,
	SPARC_INS_TA,
	SPARC_INS_TADDCCTV,
	SPARC_INS_TADDCC,
	SPARC_INS_T,
	SPARC_INS_TSUBCCTV,
	SPARC_INS_TSUBCC,
	SPARC_INS_UDIVCC,
	SPARC_INS_UDIVX,
	SPARC_INS_UDIV,
	SPARC_INS_UMAC,
	SPARC_INS_UMULCC,
	SPARC_INS_UMULXHI,
	SPARC_INS_UMUL,
	SPARC_INS_UNIMP,
	SPARC_INS_FCMPD,
	SPARC_INS_FCMPED,
	SPARC_INS_FCMPEQ,
	SPARC_INS_FCMPES,
	SPARC_INS_FCMPQ,
	SPARC_INS_FCMPS,
	SPARC_INS_WR,
	SPARC_INS_WRPR,
	SPARC_INS_XMULX,
	SPARC_INS_XMULXHI,
	SPARC_INS_XNORCC,
	SPARC_INS_XNOR,
	SPARC_INS_XORCC,
	SPARC_INS_XOR,

	// clang-format on
	// generated content <SparcGenCSInsnEnum.inc> end

	SPARC_INS_ENDING,	// <-- mark the end of the list of instructions

	SPARC_INS_ALIAS_BEGIN,
	// generated content <SparcGenCSAliasEnum.inc> begin
	// clang-format off

	SPARC_INS_ALIAS_MOVA, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSA, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDA, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_MOVRZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_BA, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BA_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BA_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BA_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BA_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BA_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_FMOVQA, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TA, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_FBA, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBA_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBA_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBA_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBA_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBA_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_BRZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_BN, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BN_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BN_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BN_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BN_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BN_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVN, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSN, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDN, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQN, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TN, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BNE, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BNE_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BNE_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BNE_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BNE_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BNE_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVNE, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSNE, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDNE, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQNE, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TNE, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BE, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BE_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BE_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BE_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BE_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BE_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVE, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSE, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDE, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQE, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TE, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BG, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BG_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BG_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BG_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BG_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BG_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVG, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSG, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDG, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQG, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TG, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BLE, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BLE_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BLE_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BLE_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BLE_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BLE_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVLE, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSLE, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDLE, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQLE, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TLE, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BGE, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BGE_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BGE_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BGE_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BGE_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BGE_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVGE, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSGE, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDGE, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQGE, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TGE, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BL, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BL_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BL_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BL_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BL_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BL_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVL, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSL, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDL, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQL, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TL, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BGU, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BGU_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BGU_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BGU_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BGU_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BGU_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVGU, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSGU, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDGU, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQGU, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TGU, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BLEU, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BLEU_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BLEU_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BLEU_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BLEU_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BLEU_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVLEU, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSLEU, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDLEU, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQLEU, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TLEU, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BCC, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BCC_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BCC_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BCC_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BCC_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BCC_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVCC, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSCC, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDCC, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQCC, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TCC, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BCS, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BCS_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BCS_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BCS_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BCS_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BCS_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVCS, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSCS, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDCS, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQCS, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TCS, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BPOS, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BPOS_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BPOS_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BPOS_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BPOS_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BPOS_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVPOS, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSPOS, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDPOS, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQPOS, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TPOS, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BNEG, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BNEG_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BNEG_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BNEG_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BNEG_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BNEG_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVNEG, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSNEG, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDNEG, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQNEG, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TNEG, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BVC, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BVC_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BVC_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BVC_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BVC_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BVC_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVVC, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSVC, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDVC, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQVC, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TVC, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_BVS, // Real instr.: SPARC_BCOND
	SPARC_INS_ALIAS_BVS_A, // Real instr.: SPARC_BCONDA
	SPARC_INS_ALIAS_BVS_PT, // Real instr.: SPARC_BPICC
	SPARC_INS_ALIAS_BVS_A_PT, // Real instr.: SPARC_BPICCA
	SPARC_INS_ALIAS_BVS_PN, // Real instr.: SPARC_BPICCNT
	SPARC_INS_ALIAS_BVS_A_PN, // Real instr.: SPARC_BPICCANT
	SPARC_INS_ALIAS_MOVVS, // Real instr.: SPARC_MOVICCrr
	SPARC_INS_ALIAS_FMOVSVS, // Real instr.: SPARC_FMOVS_ICC
	SPARC_INS_ALIAS_FMOVDVS, // Real instr.: SPARC_FMOVD_ICC
	SPARC_INS_ALIAS_FMOVQVS, // Real instr.: SPARC_FMOVQ_ICC
	SPARC_INS_ALIAS_TVS, // Real instr.: SPARC_TICCrr
	SPARC_INS_ALIAS_FBN, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBN_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBN_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBN_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBN_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBN_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBU, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBU_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBU_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBU_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBU_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBU_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVU, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSU, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDU, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQU, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBG, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBG_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBG_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBG_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBG_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBG_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBUG, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUG_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUG_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUG_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUG_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBUG_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVUG, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSUG, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDUG, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQUG, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBL, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBL_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBL_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBL_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBL_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBL_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBUL, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUL_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUL_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUL_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUL_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBUL_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVUL, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSUL, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDUL, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQUL, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBLG, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBLG_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBLG_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBLG_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBLG_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBLG_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVLG, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSLG, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDLG, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQLG, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBNE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBNE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBNE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBNE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBNE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBNE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBUE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBUE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVUE, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSUE, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDUE, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQUE, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBGE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBGE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBGE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBGE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBGE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBGE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBUGE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUGE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBUGE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUGE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBUGE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBUGE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVUGE, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSUGE, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDUGE, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQUGE, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBLE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBLE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBLE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBLE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBLE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBLE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_FBULE, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBULE_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBULE_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBULE_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBULE_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBULE_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVULE, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSULE, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDULE, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQULE, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_FBO, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBO_PT, // Real instr.: SPARC_BPFCC
	SPARC_INS_ALIAS_FBO_A, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBO_A_PT, // Real instr.: SPARC_BPFCCA
	SPARC_INS_ALIAS_FBO_PN, // Real instr.: SPARC_BPFCCNT
	SPARC_INS_ALIAS_FBO_A_PN, // Real instr.: SPARC_BPFCCANT
	SPARC_INS_ALIAS_MOVO, // Real instr.: SPARC_V9MOVFCCrr
	SPARC_INS_ALIAS_FMOVSO, // Real instr.: SPARC_V9FMOVS_FCC
	SPARC_INS_ALIAS_FMOVDO, // Real instr.: SPARC_V9FMOVD_FCC
	SPARC_INS_ALIAS_FMOVQO, // Real instr.: SPARC_V9FMOVQ_FCC
	SPARC_INS_ALIAS_BRLEZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRLEZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRLEZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRLEZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRLEZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRLEZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_MOVRLEZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSLEZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDLEZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQLEZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_BRLZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRLZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRLZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRLZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRLZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRLZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_MOVRLZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSLZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDLZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQLZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_BRNZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRNZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRNZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRNZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRNZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRNZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_MOVRNZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSNZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDNZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQNZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_BRGZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRGZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRGZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRGZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRGZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRGZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_MOVRGZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSGZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDGZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQGZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_BRGEZ, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRGEZ_PT, // Real instr.: SPARC_BPR
	SPARC_INS_ALIAS_BRGEZ_PN, // Real instr.: SPARC_BPRNT
	SPARC_INS_ALIAS_BRGEZ_A, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRGEZ_A_PT, // Real instr.: SPARC_BPRA
	SPARC_INS_ALIAS_BRGEZ_A_PN, // Real instr.: SPARC_BPRANT
	SPARC_INS_ALIAS_MOVRGEZ, // Real instr.: SPARC_MOVRrr
	SPARC_INS_ALIAS_FMOVRSGEZ, // Real instr.: SPARC_FMOVRS
	SPARC_INS_ALIAS_FMOVRDGEZ, // Real instr.: SPARC_FMOVRD
	SPARC_INS_ALIAS_FMOVRQGEZ, // Real instr.: SPARC_FMOVRQ
	SPARC_INS_ALIAS_CMP, // Real instr.: SPARC_SUBCCrr
	SPARC_INS_ALIAS_TST, // Real instr.: SPARC_ORCCrr
	SPARC_INS_ALIAS_RET, // Real instr.: SPARC_RET
	SPARC_INS_ALIAS_RETL, // Real instr.: SPARC_RETL
	SPARC_INS_ALIAS_RESTORE, // Real instr.: SPARC_RESTORErr
	SPARC_INS_ALIAS_SAVE, // Real instr.: SPARC_SAVErr
	SPARC_INS_ALIAS_CAS, // Real instr.: SPARC_CASArr
	SPARC_INS_ALIAS_CASL, // Real instr.: SPARC_CASArr
	SPARC_INS_ALIAS_CASX, // Real instr.: SPARC_CASXArr
	SPARC_INS_ALIAS_CASXL, // Real instr.: SPARC_CASXArr
	SPARC_INS_ALIAS_MOV, // Real instr.: SPARC_ORrr
	SPARC_INS_ALIAS_FCMPS, // Real instr.: SPARC_V9FCMPS
	SPARC_INS_ALIAS_FCMPD, // Real instr.: SPARC_V9FCMPD
	SPARC_INS_ALIAS_FCMPQ, // Real instr.: SPARC_V9FCMPQ
	SPARC_INS_ALIAS_FCMPES, // Real instr.: SPARC_V9FCMPES
	SPARC_INS_ALIAS_FCMPED, // Real instr.: SPARC_V9FCMPED
	SPARC_INS_ALIAS_FCMPEQ, // Real instr.: SPARC_V9FCMPEQ

	// clang-format on
	// generated content <SparcGenCSAliasEnum.inc> end

	SPARC_INS_ALIAS_CALL, // Real instr.: SPARC_INS_JMPL

	SPARC_INS_ALIAS_END,
} sparc_insn;

/// Group of SPARC instructions
typedef enum sparc_insn_group {
	// Generic groups
	// all jump instructions (conditional+direct+indirect jumps)
	SPARC_GRP_INVALID = 0, ///< = CS_GRP_INVALID = 0
	SPARC_GRP_JUMP, ///< = CS_GRP_JUMP
	SPARC_GRP_CALL, ///< = CS_GRP_CALL
	SPARC_GRP_RET, ///< = CS_GRP_RET
	SPARC_GRP_INT, ///< = CS_GRP_INT
	SPARC_GRP_IRET, ///< = CS_GRP_IRET
	SPARC_GRP_PRIVILEGE, ///< = CS_GRP_PRIVILEGE
	SPARC_GRP_BRANCH_RELATIVE, ///< = CS_GRP_BRANCH_RELATIVE

	// Architecture-specific groups
	// generated content <SparcGenCSFeatureEnum.inc> begin
	// clang-format off

	SPARC_FEATURE_IS64BIT = 128,
	SPARC_FEATURE_USESOFTMULDIV,
	SPARC_FEATURE_HASV9,
	SPARC_FEATURE_HASVIS,
	SPARC_FEATURE_HASVIS2,
	SPARC_FEATURE_HASVIS3,
	SPARC_FEATURE_HASCASA,
	SPARC_FEATURE_HASPWRPSR,

	// clang-format on
	// generated content <SparcGenCSFeatureEnum.inc> end

	SPARC_GRP_ENDING,   // <-- mark the end of the list of groups
} sparc_insn_group;

#ifdef __cplusplus
}
#endif

#endif
