//
// Syd: rock-solid application kernel
// src/kernel/ioctl.rs: ioctl(2) handler
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use libseccomp::ScmpNotifResp;
use nix::errno::Errno;

use crate::{
    hook::{SysArg, SysFlags, UNotifyEventRequest},
    kernel::syscall_path_handler,
    log_enabled,
    sandbox::{Action, Capability},
    syslog::LogLevel,
    warn,
};

#[allow(clippy::cognitive_complexity)]
pub(crate) fn sys_ioctl(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;
    let arg = req.data.args[1];

    // Lock the sandbox for read,
    // and check if the ioctl(2) request is allowlisted or denylisted.
    let cap = Capability::CAP_IOCTL;
    let sandbox = request.get_sandbox();
    let list = sandbox.ioctl_is_listed(&arg);
    let action = sandbox.default_action(cap);
    let verbose = sandbox.verbose;
    drop(sandbox); // release the read-lock.

    if list == Some(false) {
        // _ioctl_(2) request is allowlisted.
        //
        // SAFETY: ioctl is fd-only.
        return unsafe { request.continue_syscall() };
    }

    if list == Some(true) {
        // _ioctl_(2) request is denylisted.
        let filter = action == Action::Filter;

        if !filter && action >= Action::Warn && log_enabled!(LogLevel::Warn) {
            let grp = cap.to_string().to_ascii_lowercase();
            if verbose {
                warn!("ctx": "access", "cap": cap, "act": action,
                    "sys": "ioctl", "ioctl": arg,
                    "tip": format!("configure `{grp}/allow+{arg:#x}'"),
                    "req": &request);
            } else {
                warn!("ctx": "access", "cap": cap, "act": action,
                    "sys": "ioctl", "ioctl": arg,
                    "tip": format!("configure `{grp}/allow+{arg:#x}'"),
                    "pid": request.scmpreq.pid);
            }
        }

        return match action {
            Action::Allow | Action::Warn => {
                // SAFETY: ioctl is fd-only.
                unsafe { request.continue_syscall() }
            }
            Action::Filter | Action::Deny => request.fail_syscall(Errno::EACCES),
            Action::Panic => panic!(),
            Action::Exit => std::process::exit(libc::EACCES),
            action => {
                // Stop|Kill
                let _ = request.kill(action);
                request.fail_syscall(Errno::EACCES)
            }
        };
    }

    // SAFETY:
    // 1. ioctl is fd-only, so UNSAFE_CONT is ok.
    // 2. We do not pass `fsflags` which defaults to MUST_PATH.
    let argv = &[SysArg {
        dirfd: Some(0),
        flags: SysFlags::UNSAFE_CONT,
        ..Default::default()
    }];

    syscall_path_handler(request, "ioctl", argv, |path_args, request, sandbox| {
        let restrict_magiclinks = !sandbox.allow_unsafe_magiclinks();
        drop(sandbox); // release the read-lock.

        // SAFETY: SysArg has one element.
        #[allow(clippy::disallowed_methods)]
        let path = path_args.0.as_ref().unwrap();

        // Check file type.
        if let Some(typ) = path.typ.as_ref() {
            // Restriction 1: Deny block device ioctl(2) unconditionally.
            if typ.is_block_device() {
                return Ok(request.fail_syscall(Errno::EACCES));
            }

            // Restriction 2: Deny magic link ioctl(2),
            // unless trace/allow_unsafe_magiclinks:1 is set.
            if restrict_magiclinks && typ.is_magic_link() {
                return Ok(request.fail_syscall(Errno::EACCES));
            }
        } else {
            // No file type, file disappeared mid-way?
            return Ok(request.fail_syscall(Errno::ENOTTY));
        }

        // SAFETY: ioctl is fd-only.
        Ok(unsafe { request.continue_syscall() })
    })
}
