// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

package fptower

import "github.com/consensys/gnark-crypto/ecc/bls12-381/fp"

// Frobenius set z to Frobenius(x), return z
func (z *E12) Frobenius(x *E12) *E12 {
	// Algorithm 28 from https://eprint.iacr.org/2010/354.pdf (beware typos!)
	var t [6]E2

	// Frobenius acts on fp2 by conjugation
	t[0].Conjugate(&x.C0.B0)
	t[1].Conjugate(&x.C0.B1)
	t[2].Conjugate(&x.C0.B2)
	t[3].Conjugate(&x.C1.B0)
	t[4].Conjugate(&x.C1.B1)
	t[5].Conjugate(&x.C1.B2)

	t[1].MulByNonResidue1Power2(&t[1])
	t[2].MulByNonResidue1Power4(&t[2])
	t[3].MulByNonResidue1Power1(&t[3])
	t[4].MulByNonResidue1Power3(&t[4])
	t[5].MulByNonResidue1Power5(&t[5])

	z.C0.B0 = t[0]
	z.C0.B1 = t[1]
	z.C0.B2 = t[2]
	z.C1.B0 = t[3]
	z.C1.B1 = t[4]
	z.C1.B2 = t[5]

	return z
}

// FrobeniusSquare set z to Frobenius^2(x), and return z
func (z *E12) FrobeniusSquare(x *E12) *E12 {
	// Algorithm 29 from https://eprint.iacr.org/2010/354.pdf (beware typos!)

	z.C0.B0 = x.C0.B0
	z.C0.B1.MulByNonResidue2Power2(&x.C0.B1)
	z.C0.B2.MulByNonResidue2Power4(&x.C0.B2)
	z.C1.B0.MulByNonResidue2Power1(&x.C1.B0)
	z.C1.B1.MulByNonResidue2Power3(&x.C1.B1)
	z.C1.B2.MulByNonResidue2Power5(&x.C1.B2)

	return z
}

// MulByNonResidue1Power1 set z=x*(1,1)^(1*(p^1-1)/6) and return z
func (z *E2) MulByNonResidue1Power1(x *E2) *E2 {
	// (3850754370037169011952147076051364057158807420970682438676050522613628423219637725072182697113062777891589506424760,151655185184498381465642749684540099398075398968325446656007613510403227271200139370504932015952886146304766135027)
	var b = E2{
		A0: fp.Element{
			506819140503852133,
			14297063575771579155,
			10946065744702939791,
			11771194236670323182,
			2081670087578406477,
			644615147456521963,
		},
		A1: fp.Element{
			12895611875574011462,
			6359822009455181036,
			14936352902570693524,
			13914887797453940944,
			3330433690892295817,
			1229183470191017903,
		},
	}
	z.Mul(x, &b)
	return z
}

// MulByNonResidue1Power2 set z=x*(1,1)^(2*(p^1-1)/6) and return z
func (z *E2) MulByNonResidue1Power2(x *E2) *E2 {
	// (0,4002409555221667392624310435006688643935503118305586438271171395842971157480381377015405980053539358417135540939436)
	var b = E2{
		A0: fp.Element{
			0,
			0,
			0,
			0,
			0,
			0,
		},
		A1: fp.Element{
			14772873186050699377,
			6749526151121446354,
			6372666795664677781,
			10283423008382700446,
			286397964926079186,
			1796971870900422465,
		},
	}
	z.Mul(x, &b)
	return z
}

// MulByNonResidue1Power3 set z=x*(1,1)^(3*(p^1-1)/6) and return z
func (z *E2) MulByNonResidue1Power3(x *E2) *E2 {
	// (1028732146235106349975324479215795277384839936929757896155643118032610843298655225875571310552543014690878354869257,1028732146235106349975324479215795277384839936929757896155643118032610843298655225875571310552543014690878354869257)
	var b = E2{
		A0: fp.Element{
			8921533702591418330,
			15859389534032789116,
			3389114680249073393,
			15116930867080254631,
			3288288975085550621,
			1021049300055853010,
		},
		A1: fp.Element{
			8921533702591418330,
			15859389534032789116,
			3389114680249073393,
			15116930867080254631,
			3288288975085550621,
			1021049300055853010,
		},
	}
	z.Mul(x, &b)
	return z
}

// MulByNonResidue1Power4 set z=x*(1,1)^(4*(p^1-1)/6) and return z
func (z *E2) MulByNonResidue1Power4(x *E2) *E2 {
	// 4002409555221667392624310435006688643935503118305586438271171395842971157480381377015405980053539358417135540939437
	var b = fp.Element{
		9875771541238924739,
		3094855109658912213,
		5802897354862067244,
		11677019699073781796,
		1505592401347711080,
		1505729768134575418,
	}
	z.A0.Mul(&x.A0, &b)
	z.A1.Mul(&x.A1, &b)
	return z
}

// MulByNonResidue1Power5 set z=x*(1,1)^(5*(p^1-1)/6) and return z
func (z *E2) MulByNonResidue1Power5(x *E2) *E2 {
	// (877076961050607968509681729531255177986764537961432449499635504522207616027455086505066378536590128544573588734230,3125332594171059424908108096204648978570118281977575435832422631601824034463382777937621250592425535493320683825557)
	var b = E2{
		A0: fp.Element{
			9428352843095270463,
			11709709036094816655,
			14335180424952013185,
			8441381030041026197,
			5369959062663957099,
			1665664447512374973,
		},
		A1: fp.Element{
			3974078172982593132,
			8947176549131943536,
			11547238222321620130,
			17244701004083237929,
			42144715806745195,
			208134170135164893,
		},
	}
	z.Mul(x, &b)
	return z
}

// 793479390729215512621379701633421447060886740281060493010456487427281649075476305620758731620351
var nonResidue2Power1 = fp.Element{
	17076301903736715834,
	13907359434105313836,
	1063007777899403918,
	15402659025741563681,
	5125705813544623108,
	76826746747117401,
}

// MulByNonResidue2Power1 set z=x*(1,1)^(1*(p^2-1)/6) and return z
func (z *E2) MulByNonResidue2Power1(x *E2) *E2 {
	z.A0.Mul(&x.A0, &nonResidue2Power1)
	z.A1.Mul(&x.A1, &nonResidue2Power1)
	return z
}

// 793479390729215512621379701633421447060886740281060493010456487427281649075476305620758731620350
var nonResidue2Power2 = fp.Element{
	3526659474838938856,
	17562030475567847978,
	1632777218702014455,
	14009062335050482331,
	3906511377122991214,
	368068849512964448,
}

// MulByNonResidue2Power2 set z=x*(1,1)^(2*(p^2-1)/6) and return z
func (z *E2) MulByNonResidue2Power2(x *E2) *E2 {
	z.A0.Mul(&x.A0, &nonResidue2Power2)
	z.A1.Mul(&x.A1, &nonResidue2Power2)
	return z
}

// 4002409555221667393417789825735904156556882819939007885332058136124031650490837864442687629129015664037894272559786
var nonResidue2Power3 = fp.Element{
	4897101644811774638,
	3654671041462534141,
	569769440802610537,
	17053147383018470266,
	17227549637287919721,
	291242102765847046,
}

// MulByNonResidue2Power3 set z=x*(1,1)^(3*(p^2-1)/6) and return z
func (z *E2) MulByNonResidue2Power3(x *E2) *E2 {
	z.A0.Mul(&x.A0, &nonResidue2Power3)
	z.A1.Mul(&x.A1, &nonResidue2Power3)
	return z
}

// 4002409555221667392624310435006688643935503118305586438271171395842971157480381377015405980053539358417135540939436
var nonResidue2Power4 = fp.Element{
	14772873186050699377,
	6749526151121446354,
	6372666795664677781,
	10283423008382700446,
	286397964926079186,
	1796971870900422465,
}

// MulByNonResidue2Power4 set z=x*(1,1)^(4*(p^2-1)/6) and return z
func (z *E2) MulByNonResidue2Power4(x *E2) *E2 {
	z.A0.Mul(&x.A0, &nonResidue2Power4)
	z.A1.Mul(&x.A1, &nonResidue2Power4)
	return z
}

// 4002409555221667392624310435006688643935503118305586438271171395842971157480381377015405980053539358417135540939437
var nonResidue2Power5 = fp.Element{
	9875771541238924739,
	3094855109658912213,
	5802897354862067244,
	11677019699073781796,
	1505592401347711080,
	1505729768134575418,
}

// MulByNonResidue2Power5 set z=x*(1,1)^(5*(p^2-1)/6) and return z
func (z *E2) MulByNonResidue2Power5(x *E2) *E2 {
	z.A0.Mul(&x.A0, &nonResidue2Power5)
	z.A1.Mul(&x.A1, &nonResidue2Power5)
	return z
}
